Add LoadPath "../prelude".
Add LoadPath "../graph".
Add LoadPath "../ra".

Require Import ssreflect ssrfun ssrbool eqtype ssrnat seq.
Require Import fintype path finset fingraph finfun tuple.

Require Import my_ssr.
Require Import graph. 
Require Import labelling.
Require Import gen.

Set Implicit Arguments.
Unset Strict Implicit.
Import Prenex Implicits.


(** * Introduction 

     Tools to reason about randomized distributed algorithms in a generic way.  
 *)

Section general.

(** * General Case

      In this section, we define rounds, steps and monte carlo for algorithms 
      which correspond to a rewriting of states over the vertices and states 
      over locations.
      Locations could be ports or edges or anything else just expecting a 
      finType.
 *)


(** 
     We consider a graph with a set of vertices V and states over those 
     vertices of type VLabel
 *)

Variable (V: finType) (VLab: eqType).


(**  
     Locations and type of location labels
 *)

Variables (L:finType) (LLab:eqType).

(** 
     Labelling function : VState for the vertices ; LState for the locations 
 *)
Let VSt := LabelFunc V VLab.
Let LSt := LabelFunc L LLab.

(** 
     A vertex v can only change a part of the location which is 
     (WriteArea v).
     A vertex v can only have access to a part of the location which is
     (ReadArea v).
 *)
Variable WriteArea : V -> {set L}. 
Variable ReadArea : V -> {set L}. 

(** 
       Transformation of a local computation of a vertex to a global one.
       The input of Lwrite is supposed to be the writeArea of the vertex.
 *)
Variable Vwrite : VLab -> V -> VSt.
Variable Lwrite : (seq LLab) -> V -> LSt.

(** 
       Transformation of a global computation of a vertex to a local one.
       Linread gives the labels of the readArea.
       Loutread gives the labels of the writeArea.
 *)
Variable Vread : VSt -> V -> VLab.
Variable Linread : LSt -> V -> (seq LLab).
Variable Loutread : LSt -> V -> (seq LLab).

Hypothesis Vread1 : forall v w resV f,
 v != w ->
(Vread resV v) = 
(Vread (update [set w] resV f) v).

Hypothesis Lread1 : forall v w resL f,
 v != w ->
  (Loutread resL v) = 
  (Loutread (update (WriteArea w) resL f) v).

(**
     A local rule for a vertex v takes as parameters:
      the state of v ;
      the states of the writing zone ;
      the states of the reading zone.
    It gives a new state for v and new states for the writing zone.
 *)
Definition GLocT := VLab  -> (seq LLab) -> (seq LLab) -> 
 gen (VLab * seq LLab).


Section round.
(** ** Round
 *)

(** 
    Round for a randomized distributed algorithm:
    a local function is applied to all vertices which updates the global state
 *)
Fixpoint GRound (seqV: seq V) (res: VSt * LSt)
 (LocalRule : GLocT)
 : gen (VSt*LSt) :=
 match seqV with 
  |nil => Greturn _ res
  |h::t => Gbind _ _ (GRound t res LocalRule) 
            (fun s => Gbind _ _ (LocalRule (Vread res.1 h) 
                                      (Loutread res.2 h)
                                      (Linread res.2 h))
                   (fun p => Greturn _ (  (update [set h] s.1 
                                              (Vwrite p.1 h)),
                                      (update (WriteArea h) s.2 
                                              (Lwrite p.2 h))))) 
 end.

End round.

Section iterated.
(** ** Iteration of rounds
 *)


(** 
    Let LCs be e sequence of local rules, a step is the application of 
    each element in LCs to all vertices
 *)
Fixpoint GStep (LCs: seq GLocT) (seqV: seq V) (res: VSt* LSt) : gen (VSt*LSt) :=
match LCs with 
 | nil => Greturn _ res
 |a1::a2 =>  Gbind _  _ (GRound seqV res a1) 
                                  (fun y =>  GStep a2 seqV y) 
end.

(** 
    Monte Carlo: The iteration of a step n times
 *)
Fixpoint GMC (n:nat) (LCs : seq GLocT) (seqV : seq V) (res: VSt*LSt)  
 : gen(VSt*LSt) :=
 match n with 
   |O => Greturn _ res
   | S m => Gbind _ _ (GStep LCs seqV res) 
                    (fun y => GMC m LCs seqV y)
 end.   

End iterated.

End general.

Section port. 
(** * Message passing algorithm: rewriting over ports.

      In this section, we define rounds, steps and monte carlo for algorithms 
      which correspond to a rewriting of states over the vertices and states 
      over ports.
 *)

(**
    We consider a simple undirected graph with vertices in V and 
    with an edge relation Adj.
 *)
Generalizable Variables V Adj.
Context `(NG: NGraph V Adj).

(** 
    This graph is equipped with a port numbering nu: 
    each vertex see its neighbours following a predetermined order.

 *)
Variable (nu: V -> seq V). 
Hypothesis Hnu: forall (v w:V), (Adj v w) = (w \in (nu v)). 
Hypothesis Hnu2: forall (v:V), uniq (nu v).

(**
       State over vertices are of type VLab and over ports of type Plab.
       We assume that the set of labels on ports are not empty.
 *)
Variable  (VLab: eqType) (PLab: eqType).
Variable pl0:PLab. 

(** 
     Labelling functions over vertices and over ports.
     We assume that the set of ports is not empty. 
 *)  
Let Pt := (@port_finType V Adj). 
Let VSt := LabelFunc V VLab.
Let PSt := LabelFunc Pt PLab.

Variable p0: Pt. 

(** 
       Transformations of a local computation of a vertex to a global one and
       of a global computation to a global one.
 *)

Definition WriteArea (v: V) : {set Pt} :=
 outerport_set v. 

Definition Vwrite (s: VLab) (v: V) : VSt:=
  (finfun (fun x =>  s)).

Definition Pwrite  (s: seq PLab)  (v: V) : PSt :=
  (finfun (fun x=> nth pl0 s (index (sndp x) (nu v)))).

Definition Vread  (s: VSt) (v: V) : VLab :=
 (s v).

Definition Pinread (s: PSt) (v: V) : (seq PLab) :=
 map (fun x:V => s (VtoP x v p0)) (nu v). 
 
Definition Poutread (s: PSt) (v: V) : (seq PLab) :=
  map (fun (x:V) => s (VtoP v x p0))  (nu v).

Definition Vupdate (v:V) (s: VLab * seq PLab) (old: VSt) : VSt :=
 update [set v] old (Vwrite s.1 v).

Definition Pupdate (v:V) (s: VLab * seq PLab) (old: PSt) : PSt :=
  update (outerport_set v) old (Pwrite s.2 v).

Definition VPupdate (v:V) (s: VLab * seq PLab) (old: VSt*PSt) : VSt*PSt :=
 (Vupdate v s old.1, Pupdate v s old.2). 

Lemma Vupdate_1 : forall v w s old, 
 (Vupdate v s old) w = if (w == v) then s.1 else (old w).
Proof.
unfold Vupdate=>v w s old. 
by rewrite update_Plocal_iff in_set ffunE.
Qed.    

Lemma Pupdate_1 : forall u v w s old, 
Adj u w ->  
(Pupdate v s old) (VtoP u w p0) = 
(if u == v then nth pl0 s.2 (index w (nu v)) else old (VtoP u w p0)).
Proof.
unfold Pupdate,outerport_set=>u v w s old h. 
rewrite update_Plocal_iff ffunE in_set. 
by rewrite (VtoP2 _ h) (VtoP3 _ h).
Qed.

Lemma VPupdate_read_1 : forall (v w:V) (k:VLab * seq PLab) (res:VSt*PSt),
w != v ->   
(Vread (VPupdate w k res).1 v) = Vread res.1 v.
Proof. 
unfold VPupdate,Vread,Vupdate,Pupdate=>v w k res h=>/=.
rewrite update_Plocal_iff in_set eq_sym. 
move:h;case h:(w == v)=>// _.
Qed. 

Lemma VPupdate_read_5 : forall (v w:V) (k:VLab * seq PLab) (res:VSt*PSt),   
(Vread (VPupdate w k res).1 v) = if (w == v) then k.1
 else (Vread res.1 v).
Proof. 
unfold VPupdate,Vread,Vupdate,Pupdate=>v w k res=>/=.
rewrite update_Plocal_iff in_set eq_sym. 
unfold Vwrite;rewrite ffunE.
done. 
Qed. 

Lemma VPupdate_read_2 : forall (v w:V) (k:VLab*seq PLab) (res:VSt*PSt),
w != v ->   
(Poutread (VPupdate w k res).2 v) = Poutread res.2 v.
Proof. 
unfold VPupdate,Vupdate, Pupdate,Poutread=>v w k res h=>/=.
apply eq_in_map=> u hu/=.
rewrite update_Plocal_iff. 
case h1:(VtoP v u p0  \in outerport_set w)=>//. 
move:h1;rewrite -Hnu in hu;rewrite in_set (VtoP2 p0 hu) eq_sym. 
move:h;case:(w == v)=>//.
Qed.

Lemma VPupdate_read_3 : forall (v:V) (k:VLab*seq PLab) (res:VSt* PSt),
 seq.size k.2 = seq.size (nu v) ->
(Poutread (VPupdate v k res).2 v) = k.2.
Proof. 
unfold VPupdate,Vupdate,Pupdate,Poutread,Vwrite,Pwrite=>v.
have : (uniq (nu v)) by apply Hnu2. 
have : (forall w, w \in (nu v) -> Adj v w) by intro w;rewrite Hnu.
elim:(nu v);simpl;intros;first  by rewrite (size0nil H).
move:H0;case:k.2=>//=a0 l0 H0. 
rewrite update_Plocal_iff.
have hva: (Adj v a) by apply x;rewrite in_cons eq_refl. 
case h1:(VtoP v a p0  \in outerport_set v);last first.
 by move:h1;rewrite in_set (VtoP2 _ hva) eq_refl=>//.
rewrite ffunE (VtoP3 _ hva) eq_refl=>/=.
move/andP:x0=>[x0 x0'].
apply f_equal;set p:= (a0::l0);change l0 with (k.1,l0).2.  
rewrite -(H _ _ _ res);auto;last first.
  by intros;apply x;rewrite in_cons H1 orbT;auto.
apply eq_in_map;intros z hz;unfold p;clear p.
do 2 rewrite update_Plocal_iff.
have hz':Adj v z by  apply x;rewrite in_cons hz orbT.
case hx':( VtoP v z p0 \in outerport_set v)=>//.
do 2 rewrite ffunE/=.
rewrite VtoP3=>//. 
case hax: (a == z)=>//=;auto. 
move/eqP:hax=>hax;subst a.
by rewrite hz in x0.
Qed. 


Lemma VPupdate_read_4 : forall (v:V) (k:VLab * seq PLab) (res:VSt*PSt),
(Poutread (VPupdate v k res).2 v)  = 
(take (seq.size (nu v)) (k.2++(nseq (seq.size (nu v)) pl0))).
Proof. 
unfold VPupdate,Vupdate,Pupdate,Poutread,Vwrite,Pwrite=>v.
have : (uniq (nu v)) by apply Hnu2. 
have : (forall w, w \in (nu v) -> Adj v w) by intro w;rewrite Hnu.
elim:(nu v);intros. 
 by rewrite take0.
have x':   forall w : V, w \in l -> Adj v w.
 intros w hw; apply x;by rewrite in_cons hw orbT. 
have hva: (Adj v a) by apply x;rewrite in_cons eq_refl. 
case h1:(VtoP v a p0  \in outerport_set v);last by move:h1;
 rewrite in_set (VtoP2 _ hva) eq_refl=>//.
rewrite cons_uniq in x0;move/andP:x0=>[x0 x0'].
rewrite map_cons ffunE h1 ffunE (VtoP3 _ hva) index_head nth0.
case:k.2=>/=.
 apply f_equal;have hyp := (H _  x0'(k.1,nil) res);simpl in hyp.
 rewrite -hyp=>//;clear hyp.
 apply eq_in_map=>z hz; repeat rewrite ffunE. 
 case h:( VtoP v z p0 \in outerport_set v)=>//=.
 by do 2 rewrite nth_nil.
intros;apply f_equal. 
rewrite -cat_rcons.  
have h:= (H x' x0' (k.1,rcons l0 pl0) res);simpl in h. 
rewrite -h;clear h.
apply eq_in_map. 
intros z hz;repeat rewrite ffunE. 
case h:(VtoP v z p0 \in outerport_set v)=>//=.
case hax: (a == (sndp (VtoP v z p0))).
 rewrite (VtoP3 _ (x'  z hz)) in hax.
 by move/eqP:hax=>hax;rewrite -hax in hz;rewrite hz in x0.
simpl;rewrite nth_rcons. 
case h':(index (sndp (VtoP v z p0)) l < size l0)%nat=>//=. 
rewrite nth_default. 
 by case:(index (sndp (VtoP v z p0)) l == size l0). 
by rewrite leqNgt h'.
Qed.

Lemma VPupdate_read_6 : forall (v w:V) (k:VLab * seq PLab) (res:VSt*PSt),
(Poutread (VPupdate w k res).2 v)  = if (w == v) then
(take (seq.size (nu v)) (k.2++(nseq (seq.size (nu v)) pl0))) 
else  Poutread res.2 v.
Proof.
move=>v w k res. 
case h:(w ==v). 
 move/eqP:h=>h;subst w;apply VPupdate_read_4. 
apply VPupdate_read_2.
by rewrite h. 
Qed.

Lemma VPupdate_read_7 : forall (v:V) (k:VLab*seq PLab) (res:VSt*PSt),
(Pinread (VPupdate v k res).2 v) = Pinread res.2 v.
Proof. 
unfold VPupdate,Vupdate, Pupdate,Pinread=>v k res=>/=.
apply eq_in_map=> u hu/=.
rewrite update_Plocal_iff. 
case h1:(VtoP u v p0  \in outerport_set v)=>//.
move:h1;rewrite -Hnu gsym in hu;rewrite in_set (VtoP2 p0 hu) eq_sym.
case hu':(v == u)=>//. move/eqP:hu'=>hu'. subst u. 
move:hu. rewrite grefl. done.  
Qed.


Lemma VPupdate_1 : forall v w k k' x, w != v ->
 VPupdate w k (VPupdate v k' x) =  VPupdate v k' (VPupdate w k x).
Proof. 
unfold VPupdate,Vupdate,Pupdate,Vwrite,Pwrite=>v w k k' x hwv.
rewrite update_Pcomm.
 rewrite (@update_Pcomm _ _ (outerport_set w))=>//.
 apply disjoint_outerport=>//.   
rewrite -setI_eq0 -setIdE. 
case h:  ([set x0 in [set v] | x0 == w] == set0)%B=>//.
rewrite -cards_eq0 in h;move/eqP:h=>h.
destruct h;apply eq_card0;move=>x0.
rewrite in_set in_set  inE;case h:(x0 == v);auto.
move/eqP:h=>->;move:hwv;rewrite eq_sym;case:(v==w)=>//.
Qed.


Section round.
(** ** Round
 *)

Let LocPT := GLocT VLab PLab.

(** 
    Round for a randomized distributed algorithm:
    a local function is applied to all vertices which updates the global state
 *)
Definition GPRound (seqV: seq V)(res: VSt*PSt)(LC:LocPT) : gen (VSt*PSt):=
 GRound WriteArea Vwrite Pwrite Vread Pinread Poutread seqV res LC.

End round.

Section iterated.
(** ** Iteration of rounds
 *)

Let LocPT := GLocT VLab PLab.


(** 
    Let LCs be e sequence of local rules, a step is the application of 
    each element in LCs to all vertices
 *)
Definition GPStep (LCs : seq LocPT)(seqV : seq V)(res: VSt*PSt):gen(VSt*PSt):=
 GStep WriteArea Vwrite Pwrite Vread Pinread Poutread  LCs seqV res.

(** 
    Monte Carlo: The iteration of a step n times
 *)
Definition GPMC (n:nat)(LCs:seq LocPT)(seqV:seq V)(res:VSt*PSt):gen(VSt*PSt):=
GMC WriteArea Vwrite Pwrite Vread Pinread Poutread n LCs seqV res.

End iterated. 

End port. 


