Add LoadPath "../prelude".
Add LoadPath "../graph".
Add LoadPath "../ra".

Require Import ssreflect ssrfun ssrbool eqtype ssrnat seq.
Require Import fintype path finset fingraph finfun tuple.

Require Import Ensembles.

Require Import graph.
Require Import labelling. 
Require Import gen.
Require Import setSem. 
Require Import rdaTool_gen. 

Set Implicit Arguments.
Unset Strict Implicit.
Import Prenex Implicits.



(** * Introduction
      This files describes the specification of any handshake problem
 *) 

Section hsSpec.

(** * Specification of the handshake problem in a global view 
    *)

(** 
    V: set of vertices.
    Adj: edge relation
    NG: undirected non-loop graph.
 *)

Generalizable Variables V Adj.
Context `(NG: NGraph V Adj).

(** 
    synch: type of function describing the other vertex 
           with which a vertex is in handshake
 *)

Definition synch := V -> option V.

(** 
    synchAdj s: each pairs in a handshake are adjacent
 *) 

Definition synchAdj (s: synch) := forall (v:V),
 match (s v) with
  |Some w => Adj v w
  |_ => true
 end.

(** 
    synchSym s: each member of a pairs in a handshake are 
                in a handshake with the other
 *)

Definition synchSym (s: synch) := forall (v:V),
 match (s v) with
  |Some w => (s w) == Some v
  |_ => true
 end.

(** 
   partialMatching s: s is adj and sym
 *)
Definition matching (s: synch) := 
 (synchAdj s) /\ (synchSym s).

(** 
   hsBetween s v w: there is a handshake between v and w
 *)

Definition hsBetween (s: synch) (v w:V) :=
 (Adj v w) && (s v == Some w) && (s w == Some v). 

(** 
    hsExists, there exists v and w such that they are in a handshake
 *)

Definition hsExists (s: synch) := exists v w,
 hsBetween s v w. 

End hsSpec.

(** * In a common graph: Definitions
 *)

Section commonGraph.

(** 
    V: set of vertices of the graph. 
    Adj: edge relation of the graph.
    NG: undirected non-loop graph.
    VLabel: type of labels on vertices.
    LLabel: type of labels on ports.
    vunit, lunit: default values.
    LR lv slp: local computation from a local view.
 *)

Generalizable Variables V Adj.
Context `(NG: NGraph V Adj).

Variables (VLabel: eqType) (PLabel:eqType).

Variables (vunit:VLabel) (lunit:PLabel).
 
Variable LR : seq(VLabel  -> (seq PLabel) ->(seq PLabel) ->
  gen (VLabel * seq PLabel)).

Let VSt := LabelFunc V VLabel.
Let PSt := LabelFunc (@port_finType V Adj) PLabel.

Variable p0 : (@port_finType V Adj). 

Variable (nu:V -> seq V). 
Hypothesis Hnu : forall (v w:V), (Adj v w) = (w \in (nu v)).
Hypothesis Hnu2: forall (v :V), uniq (nu v). 

Definition UniformView (s: VSt* PSt) :=
 forall v w, seq.size (nu v) = seq.size (nu w) -> 
             (Vread s.1 v)= (Vread s.1 w) /\
             (Pinread nu p0 s.2 v) = (Pinread nu p0 s.2 w) /\
             (Poutread nu p0 s.2 v) = (Poutread nu p0 s.2 w).

Definition Uniform (s: VSt* PSt) :=
 (forall v1 v2, (s.1 v1)= (s.1 v2)) /\
 (forall p1 p2, (s.2 p1) = (s.2 p2)).

Lemma uniformUniformView : forall (s: VSt* PSt),
Uniform s -> UniformView s.
Proof.
unfold Uniform,UniformView;intros s  [H1 H2] v w;split. 
 apply H1.
unfold Pinread,Poutread.  
move:H.
set l:= (nu v).  
elim:(nu w) l=>//=.
  intros. apply size0nil in H. rewrite H. done. 
intros. case:l0 H0=>//=;intros. apply eq_add_S in H0. 
apply H in H0. destruct H0. rewrite H0 H3. 
rewrite (H2 (VtoP a0 v p0) (VtoP a w p0)).
rewrite (H2 (VtoP v a0 p0) (VtoP w a p0)).
done. 
Qed. 
 

(** 
    nextState sigma pSeq: a round over sigma with the pSeq order
 *)   

Definition nextState (sV: seq V)(sigma:VSt* PSt):=
GPStep nu lunit p0 LR sV sigma.

(** 
   We assume there is a function, hsPort, wich tells from a local view 
   if there is a handshake for a vertex v and on which port
 *)

Variable hsPort : VLabel -> seq PLabel -> seq PLabel -> option nat.

Definition hsPortR (sigma: VSt *PSt)  (v:V)  :=  
 (hsPort (Vread sigma.1 v) (Poutread nu p0 sigma.2 v) (Pinread nu p0 sigma.2 v)) .

Hypothesis hsp1 : forall (sigma: VSt *PSt)  (v:V) i, 
  (hsPortR sigma  v)  = Some i -> 
  i < (deg Gr v). 

(**
   assNeigh v: returns None if v is not in handshake 
               or Some w, if w is in handshake with w
 *)
  
Definition assNeigh (v: V) (sigma:VSt * PSt)  : (option V) :=
 match (hsPortR sigma v) with 
  |Some i => Some (nth v (nu v) i)
  |_ => None 
 end.

(** 
   consistent: hsPort is symmetrical
 *)

Definition consistent (sigma: VSt * PSt) :=
 synchSym (fun v => assNeigh v sigma).

(**
   hsEventually: there exists a state where there is at least one handshake
                 and which is reachable from the initState
 *)
Definition hsEventually initS sV :=
 exists sigma,
    reachFrom _ (nextState sV) initS sigma /\
     (@hsExists _ Adj (fun v => assNeigh v sigma )).


(** Lemmas
 *)
Lemma assNeigh1 : forall v w s,
 uniq (nu v) -> size (nu v) = deg Gr v -> 
 assNeigh v s  = Some w -> 
 hsPortR s  v = Some (index w (nu v)).
Proof.
unfold assNeigh;move=>v w s  h h'.
move:(@hsp1 s  v). 
case:(hsPortR s v)=>//.
move=>a hsP0 H;injection H=><-;apply f_equal. 
symmetry; apply index_uniq=>//.
rewrite h';by apply (hsP0 a).
Qed.

End commonGraph.

Section specAlg.

(** 
    V: set of vertices of the graph. 
    Adj: edge relation of the graph.
    NG: undirected non-loop graph.
    VLabel: type of labels on vertices.
    LLabel: type of labels on ports.
    vunit, lunit: default values.
    LR lv slp: local computation from a local view.
 *)

(*Generalizable Variables V Adj.
Context `(NG: NGraph V Adj).
*)
Variables (VLabel: eqType) (PLabel:eqType).

Variables (vl0:VLabel) (pl0:PLabel).
 
Let State (V:finType) (Adj:rel V) := Datatypes.prod (LabelFunc V VLabel)  
 ( LabelFunc (@port_finType V Adj) PLabel). 

Let pfT  (V:finType) (Adj:rel V) := (@port_finType V Adj).  

Record hsAlgo :=
{
(** Local rules *)
HsR : seq(VLabel  -> (seq PLabel) ->(seq PLabel) ->
  gen (VLabel * seq PLabel));

(** Handshake function *)
HsP : VLabel -> seq PLabel -> seq PLabel -> option nat;


(** Initial state *)
HsI : forall (V:finType) (Adj: rel V)(Gr:Graph Adj)(NG: NGraph Gr), State Adj;

HsI1:  forall (V:finType) (Adj: rel V) (Gr:Graph Adj)(NG: NGraph Gr) (nu:V -> seq V)  
                      (Hnu: forall (v w:V), (Adj v w) = (w \in (nu v)))  (Hnu2: forall (v :V), uniq (nu v)) 
                      (p0: pfT Adj), 
           consistent  p0 nu HsP (HsI NG); 

HsI2 :  forall (V:finType) (Adj: rel V) (Gr:Graph Adj) (NG: NGraph Gr)(nu:V -> seq V) 
                       (Hnu: forall (v w:V), (Adj v w) = (w \in (nu v)))  (Hnu2: forall (v :V), uniq (nu v)) ,
             Uniform (HsI NG);

HsP1 : forall (V:finType) (Adj: rel V) (Gr:Graph Adj) (NG: NGraph Gr)(nu:V -> seq V) 
 (Hnu: forall (v w:V), (Adj v w) = (w \in (nu v)))  (Hnu2: forall (v :V), uniq (nu v)) 
 (p0: pfT Adj) (s:State Adj) (v:V) (i:nat),
 (hsPortR p0 nu HsP s v)  = Some i -> i < (deg Gr v);


HsRind: forall (V:finType) (Adj: rel V) (Gr:Graph Adj) (NG: NGraph Gr)(nu:V -> seq V)  
                (Hnu: forall (v w:V), (Adj v w) = (w \in (nu v)))  (Hnu2: forall (v :V), uniq (nu v)) 
                (p0: pfT Adj),
        Stable _ (fun s => consistent p0 nu HsP s)  (nextState pl0 HsR p0 nu (enum V))
}.

Definition hsRealisation (A: hsAlgo) := 
 forall (V:finType) (Adj: rel V) (Gr:Graph Adj)(NG: NGraph Gr)(nu:V -> seq V) 
(Hnu: forall (v w:V), (Adj v w) = (w \in (nu v)))  (Hnu2: forall (v :V), uniq (nu v)) 
(p0: pfT Adj),
 hsEventually pl0 (HsR A) p0 nu (HsP A) (HsI A NG) (enum V).

Variable A: hsAlgo.
Hypothesis Aok : hsRealisation A. 

Fixpoint Adet (l:seq(VLabel  -> (seq PLabel) ->(seq PLabel) ->
  gen (VLabel * seq PLabel))) := 
match l with
 |nil => True 
 |t::q => (forall lv lp1 lp2, Deterministic (t lv lp1 lp2)) /\ (Adet q)
end. 

End specAlg.


(* hsAlgo :
    Record hsAlgo (VLabel PLabel : eqType) (pl0 : PLabel) : Type := Build_hsAlgo
  { HsR : seq
            (VLabel -> seq PLabel -> seq PLabel -> gen (VLabel * seq PLabel));
    HsP : VLabel -> seq PLabel -> seq PLabel -> option nat;
    HsI : forall (V : finType) (Adj : rel V),
          (fun (V0 : finType) (Adj0 : rel V0) =>
           (LabelFunc V0 VLabel * LabelFunc port_finType PLabel)%type) V Adj;
    HsI1 : forall (V : finType) (Adj : rel V) (nu : V -> seq V)
             (p0 : (fun V0 : finType => [eta port_finType]) V Adj),
           consistent p0 HsP (HsI V Adj) nu;
    HsI2 : forall (V : finType) (Adj : rel V) (nu : V -> seq V),
           Uniform (HsI V Adj) nu;
    HsP1 : forall (V : finType) (Adj : rel V) (nu : V -> seq V)
             (Gr : Graph Adj)
             (p0 : (fun V0 : finType => [eta port_finType]) V Adj)
             (s : (fun (V0 : finType) (Adj0 : rel V0) =>
                   (LabelFunc V0 VLabel * LabelFunc port_finType PLabel)%type)
                    V Adj) (v : V) (i : nat),
           hsPortR p0 HsP s nu v = Some i -> i < deg Gr v;
    HsRind : forall (V : finType) (Adj : rel V) (nu : V -> seq V)
               (p0 : (fun V0 : finType => [eta port_finType]) V Adj),
             Stable (LabelFunc V VLabel * LabelFunc port_finType PLabel)
               ((consistent p0 HsP)^~ nu) (nextState pl0 HsR p0 nu (enum V)) }
 *)

