Require Import ssreflect ssrfun ssrbool eqtype ssrnat seq.
Require Import fintype path finset fingraph finfun bigop choice tuple.

Add LoadPath "../prelude".
Require Import my_ssr.

Set Implicit Arguments.
Unset Strict Implicit.
Import Prenex Implicits.


(** * Introduction
    This file develops the theory of finite graph represented by an edge        
    relation over a finType V. 
 *)
 

(** * Definitions: Graph
 *)


(** 
     V: set of vertices of the graph  
     Adj: edge relation of the graph
 *)

Class Graph {V:finType} (Adj: rel V).

Section Graph.

Generalizable Variables V Adj.
Context `(G: Graph V Adj).


(** 
      Nb_enum G v: the ordered sequence of the neighbours of v
 *)

Definition Nb_enum (G: Graph Adj)(v: V) : seq  V := 
 enum (Adj v).


(** 
     deg G v: the degree of v, i.e. the number of neighbours it has
 *)

Definition deg (G: Graph Adj) (v: V) : nat := 
 seq.size (Nb_enum G v).


(** 
      nb_id G v w: the index of w in the sequence Nb_enum G v.
                 w is said to be (nb_id v w)th neighbour of v                  
 *)

Definition nb_id (G: Graph Adj) (v w: V) : nat := 
 index w (Nb_enum G v).


(** 
    edge_finType is the finType containing the set of edges.
    fste is the first member of the edge.
    snde is the second one.
 *) 

Record edge : Type := 
 Edge {edgeVal : (Datatypes.prod V V); 
                 EdgeValP : Adj edgeVal.1 edgeVal.2 &&
               ((enum_rank edgeVal.1) < (enum_rank edgeVal.2))%nat}.

Canonical edge_subType := Eval hnf in [subType for edgeVal by edge_rect].
Definition edge_eqMixin := Eval hnf in [eqMixin of edge by <:].
Canonical edge_eqType := Eval hnf in EqType edge edge_eqMixin.
Definition edge_choiceMixin := [choiceMixin of edge by <:].
Canonical edge_choiceType :=
  Eval hnf in ChoiceType edge edge_choiceMixin.
Definition edge_countMixin := [countMixin of edge by <:].
Canonical edge_countType := 
  Eval hnf in CountType edge edge_countMixin.
Canonical edge_subCountType := [subCountType of edge].
Definition edge_finMixin := [finMixin of edge by <:].
Canonical edge_finType := Eval hnf in FinType edge edge_finMixin.

Definition fste (x: edge) := (edgeVal x).1.
Definition snde (x: edge) := (edgeVal x).2.


(** 
    port_finType is the finType containing the set of ports.
    fstp is the first member of the port.
    sndp is the second one.
 *) 

Record port  := 
 Port {pval : (Datatypes.prod V V); PvalP : Adj pval.1 pval.2}.

Canonical port_subType := Eval hnf in [subType for pval by port_rect].
Definition port_eqMixin := Eval hnf in [eqMixin of port by <:].
Canonical port_eqType := Eval hnf in EqType port port_eqMixin.
Definition port_choiceMixin := [choiceMixin of port by <:].
Canonical port_choiceType :=
  Eval hnf in ChoiceType port port_choiceMixin.
Definition port_countMixin := [countMixin of port by <:].
Canonical port_countType := 
  Eval hnf in CountType port port_countMixin.
Canonical port_subCountType := [subCountType of port].
Definition port_finMixin := [finMixin of port by <:].
Canonical port_finType := Eval hnf in FinType port port_finMixin.

Definition fstp (x: port) := (pval x).1.
Definition sndp (x: port) := (pval x).2.


(** 
    outerport_set v: the set of ports whose first member is equal to v. 
 *)

Definition outerport_set (v:V) := 
[set x:port|  (fstp x) == v].


(** 
      outerport_list v: the default sequence of ports corresponding to (outerport_set v).
 *)

Definition outerport_list (v:V) :=
enum (outerport_set v).

(** 
    innerport_set v: the set of ports whose second member is equal to v.
 *)

Definition innerport_set (v:V) := 
[set x:port|  (sndp x) == v].


(** innerport_list v: the default sequence of ports corresponding to (innerport_set v).
 *)

Definition innerport_list (v:V) :=
enum (innerport_set v).


(** 
      port_id G v w: the index of the port (v,w) in the sequence 
      (outerport_list v).
                 (v,w) is said to be (port_id v w)th port linked to v
 *)

Definition port_id (G: Graph Adj) (v w: V) : nat := 
 find  (fun x => sndp x == w) (outerport_list v).

(** 
    VtoP v w p0: returns a port made with v and w if they are adjacent
                         p0 otherwise
 *)
Definition VtoP (v w : V) (p0 : port_finType) : port_finType := 
 odflt p0 (insub (v, w)).



(** * Lemmas: Graph
 *)

(** ** deg 
 *)

Lemma deg_index_lt : forall (v w: V), 
  Adj v w = (index w (Nb_enum G v) < deg G v).
Proof.
move => v w;by rewrite index_mem mem_enum.
Qed.

Lemma deg_zero : forall (v:V),
 deg G v = 0 <-> 
 forall (w:V), ~Adj v w.
Proof.
move=>v;split;unfold deg,Nb_enum. 
 move=>h w;case h1:(Adj v w)=>//;apply size0nil in h.
 have : w \in enum (Adj v) by simpl; rewrite mem_enum. 
 by rewrite h.  
move=>h;rewrite -cardE -(cards0 V);apply eq_card. 
move=> x;have h':= (h x).
rewrite in_set0;case h'': (x \in Adj v)=>//.
Qed.


Lemma deg_card1 : forall (v:V), irreflexive Adj ->
 deg G v <= #|V|.-1.
Proof.
unfold deg,Nb_enum;move=>v hirr.
destruct (ltngtP (size (enum (Adj v))) (#|V|.-1));last by rewrite e.
 by rewrite leq_eqVlt i Bool.orb_true_r.
have h := (subset_predT (Adj v));apply subset_leq_card in h. 
rewrite -cardE in i;rewrite prednK in i;last first. 
 case h':(0<#|V|)=>//;move/card_gt0P:h'=>h';destruct h';by exists v.
have h' : (#|Adj v| == #|V|) by rewrite eqn_leq h i=>//.
clear h i;move/eqP:h'=>h';apply subset_cardP  in h'.
have h:=(subset_predT (Adj v)); move/h':h=>h.
have h'':(v \in V) by trivial.
rewrite -h in h''.
have := (irrefl_mem _  _ v hirr);by rewrite h''. 
Qed.



(** ** nb_id 
 *)

Lemma nb_id_lt: forall (v w:V) (x:nat), 
 ~~(x < deg G v)%nat -> ~~((nb_id G v w == x) && Adj v w).
Proof. 
move=>v w x h;unfold nb_id,Nb_enum.
rewrite deg_index_lt;case hvw:(index w (enum (Adj v))==x)=>//.
by move/eqP:hvw=>->.  
Qed.  

Lemma nb_id_index_lt: forall (v w:V), 
 Adj v w = (nb_id G v w < deg G v)%nat.
Proof.
move=>v w;rewrite deg_index_lt=>//.
Qed.

Lemma deg_exists : forall (v:V) (x:nat), 
 x < deg G v -> exists w, Adj v w /\ nb_id G v w = x. 
Proof.
move=>v x h.
have h1:(forall t,t\in(enum (Adj v))->Adj v t) by move=>t;rewrite mem_enum.
have h2:uniq (enum (Adj v)) by apply enum_uniq. 
move:h1 h2 x h;unfold nb_id,deg,Nb_enum;elim: (enum (Adj v))=>//=.
move=>t s hind h1;move/andP=>[h2 h3] x h4;case hx:(x == 0). 
 move/eqP:hx=>->;exists t;rewrite eq_refl;split=>//.
 apply h1;rewrite in_cons eq_refl =>//.
have h4': x.-1 < size s.
 rewrite -(ltn_add2r 1);rewrite addn1 addn1 prednK=>//;by apply neq0_lt0n.
have h5:(exists w, Adj v w /\ index w s = x.-1).
 apply hind=>//.
 move=>t' ht;apply h1;rewrite in_cons ht Bool.orb_true_r=>//.
destruct h5 as [w [hw1 hw2]];exists w;split;auto. 
case hwt:(t==w);last by rewrite hw2 prednK=>//;apply  neq0_lt0n.
move:h2;rewrite -index_mem;move/eqP:hwt hw2->;move->.
by rewrite h4'.
Qed.

Lemma nb_id_e : forall u v v', Adj u v ->
 nb_id G u v == nb_id G u v' -> v == v'.
Proof. 
move=>u v v'. rewrite nb_id_index_lt. 
unfold nb_id,deg. elim:(Nb_enum G u)=>//=.
move=>t s  hind.
case ht: (t == v). 
 case ht':(t == v')=>//. 
  move/eqP:ht<-. move/eqP:ht'<-=>//.
 case ht':(t == v')=>//.
Qed. 


(** **  edge 
 *)

Lemma edge_fs : forall (e:edge),
 fste e != snde e.
Proof.
move=>e;unfold fste,snde.
destruct e=>/=.
move/andP:EdgeValP0=>[h1 h2].
case h:(edgeVal0.1 == edgeVal0.2)=>//.
move/eqP:h h2->;by rewrite ltnn.
Qed.

Lemma edge_fste_snde : forall (e:edge),
 Adj (fste e)  (snde e). 
Proof. 
move=>e;unfold fste,snde.
destruct e=>/=.
by move/andP:EdgeValP0=>[h1 h2].
Qed.

Lemma edge_eq : forall e1 e2,
 edgeVal e1 = edgeVal e2 -> 
 Adj (edgeVal e1).1 (edgeVal e1).2 &&
              (enum_rank (edgeVal e1).1 < enum_rank (edgeVal e1).2)%nat =
 Adj (edgeVal e1).1 (edgeVal e1).2 &&
              (enum_rank (edgeVal e1).1 < enum_rank (edgeVal e1).2)%nat ->
 e1 = e2.
Proof.
destruct e1,e2;simpl;move=>H1 H2.
subst edgeVal0.
replace EdgeValP1 with EdgeValP0=>//.
apply Eqdep_dec.UIP_dec, Bool.bool_dec. 
Qed.

Lemma edge_nth_neq1 : forall k i e,
 (k < i < #|edge_finType|)%nat -> 
 (fste (nth e (enum edge_finType) k) == 
  fste (nth e (enum edge_finType) i))%B   ->
 snde (nth e (enum edge_finType) k) != 
   snde (nth e (enum edge_finType) i).
Proof.
unfold fste,snde;move=>k i e hk hb.
case hb':((edgeVal (nth e (enum edge_finType) k)).2 ==
  (edgeVal (nth e (enum edge_finType) i)).2)=>//.
move:hk;rewrite cardE;case hk1:(k<i)%nat=>//;
 case hk2:(i<(seq.size (enum edge_finType)))%nat=>//.
have hyp : ((nth e (enum edge_finType) k) == 
 (nth e (enum edge_finType) i) = false).  
 rewrite nth_uniq=>//;last by apply enum_uniq.
 +case hk':(k == i)=>//;move/eqP:hk' hk1->;by rewrite ltnn.
 +apply (@ltn_trans i)=>//. 
move=>_;move/eqP:hyp=>hyp;move/eqP:hb=>hb;move/eqP:hb'=>hb'. 
destruct hyp;apply edge_eq=>//.
apply (injective_projections _ _ hb hb'). 
Qed.


Lemma edge_nth_neq2 : forall k i e,
 (k < i < #|edge_finType|)%nat -> 
 (fste (nth e (enum edge_finType) k) == 
  snde (nth e (enum edge_finType) i))%B   ->
 snde (nth e (enum edge_finType) k) != 
   fste (nth e (enum edge_finType) i).
Proof.
unfold fste,snde;move=>k i e hk hb.
case hb':((edgeVal (nth e (enum edge_finType) k)).2 ==
  (edgeVal (nth e (enum edge_finType) i)).1)=>//.
move:hk;rewrite cardE;case hk1:(k<i)%nat=>//;
 case hk2:(i<(seq.size (enum edge_finType)))%nat=>//.
move=>_;move/eqP:hb=>hb;move/eqP:hb'=>hb'.
move/andP:(EdgeValP (nth e (enum edge_finType) k))=>[_ hyp].  
rewrite hb hb' in hyp.
move/andP:(EdgeValP (nth e (enum edge_finType) i))=>[_ hyp'].
by rewrite ltnNge leq_eqVlt hyp' Bool.orb_true_r in hyp.
Qed.

Lemma edge_nth_neq3 : forall k i e,
 (k < i < #|edge_finType|)%nat -> 
 (snde (nth e (enum edge_finType) k) == 
  fste (nth e (enum edge_finType) i))%B   ->
 fste (nth e (enum edge_finType) k) != 
   snde (nth e (enum edge_finType) i).
Proof.
unfold fste,snde;move=>k i e hk hb.
case hb':((edgeVal (nth e (enum edge_finType) k)).1 ==
  (edgeVal (nth e (enum edge_finType) i)).2)=>//.
move:hk;rewrite cardE;case hk1:(k<i)%nat=>//;
 case hk2:(i<(seq.size (enum edge_finType)))%nat=>//.
move=>_;move/eqP:hb=>hb;move/eqP:hb'=>hb'.
move/andP:(EdgeValP (nth e (enum edge_finType) k))=>[_ hyp].  
rewrite hb hb' in hyp.
move/andP:(EdgeValP (nth e (enum edge_finType) i))=>[_ hyp'].
by rewrite ltnNge leq_eqVlt hyp' Bool.orb_true_r in hyp.
Qed.

Lemma edge_nth_neq4 : forall k i e,
 (k < i < #|edge_finType|)%nat -> 
 (snde (nth e (enum edge_finType) k) == 
  snde (nth e (enum edge_finType) i))%B   ->
 fste (nth e (enum edge_finType) k) != 
   fste (nth e (enum edge_finType) i).
Proof.
unfold fste,snde;move=>k i e hk hb.
case hb':((edgeVal (nth e (enum edge_finType) k)).1 ==
  (edgeVal (nth e (enum edge_finType) i)).1)=>//.
move:hk;rewrite cardE;case hk1:(k<i)%nat=>//;
 case hk2:(i<(seq.size (enum edge_finType)))%nat=>//.
have hyp : ((nth e (enum edge_finType) k) == 
 (nth e (enum edge_finType) i) = false).  
 rewrite nth_uniq=>//;last by apply enum_uniq.
 +case hk':(k == i)=>//;move/eqP:hk' hk1->;by rewrite ltnn.
 +apply (@ltn_trans i)=>//. 
move=>_;move/eqP:hyp=>hyp;move/eqP:hb=>hb;move/eqP:hb'=>hb'. 
destruct hyp;apply edge_eq=>//.
apply (injective_projections _ _ hb' hb). 
Qed.

Lemma edge_in_V1 : forall e:edge_finType, 
 (fste e) \in (enum V).
Proof.
move=>e;by rewrite mem_enum. 
Qed.

Lemma edge_in_V2 : forall e:edge_finType,
 (snde e) \in (enum V). 
Proof.
move=>e;by rewrite mem_enum.
Qed.

(** **  Port
 *)

Definition EtoP1 (e : edge_finType) : port_finType := 
Port  (edge_fste_snde e). 

Lemma port_eq : forall p1 p2,
 pval p1 = pval p2 -> 
 Adj (pval p1).1 (pval p1).2 =
 Adj (pval p1).1 (pval p1).2 ->
 p1 = p2.
Proof.
destruct p1,p2;simpl;move=>H1 H2.
subst pval0.
replace PvalP1 with PvalP0=>//.
apply Eqdep_dec.UIP_dec, Bool.bool_dec. 
Qed.

Lemma VtoP1 p p0 : VtoP (fstp p) (sndp p) p0 = p.
Proof.
rewrite /VtoP.
case: insubP; case=> /=;last first.
 by unfold fstp,sndp; rewrite (PvalP p).
move=> p' P1 P2 P3;apply port_eq=>//=.
rewrite P3;unfold fstp, sndp; destruct p=>//=.
by destruct (pval0).
Qed.    

Lemma VtoP2 : forall v w p0, Adj v w ->
 fstp (VtoP v w p0) = v.
Proof. 
rewrite /VtoP=> v w e0 h1.
case:insubP=>/= [u hu1 hu2 | hu];last first.
 by rewrite h1 in hu.
move:hu2;unfold fstp;destruct u=>/=;destruct pval0=>/=. 
by move=>H;injection H=>_ ->.
Qed. 

Lemma VtoP3 : forall v w p0, Adj v w ->
 sndp (VtoP v w p0) = w.
Proof. 
rewrite /VtoP=> v w p0 h1.
case:insubP=>/= [u hu1 hu2 | hu];last first.
 by rewrite h1 in hu.
move:hu2;unfold sndp;destruct u=>/=;destruct pval0=>/=. 
by move=>H;injection H=> -> _.
Qed.



Lemma disjoint_outerport : forall v w, v != w ->
  [disjoint outerport_set v & outerport_set w].
Proof.
move=>v w hvw;unfold outerport_set. 
rewrite -setI_eq0 -setIdE. 
case h:  ([set x in [set x0 |fstp x0 == v] | fstp x == w] == set0)%B=>//.
rewrite -cards_eq0 in h;move/eqP:h=>h.
destruct h;apply eq_card0;move=>x0.
rewrite in_set in_set  inE. 
case h:(fstp x0 == v);auto.
move/eqP:h=>->;move:hvw;case:(v==w)=>//.
Qed. 

End Graph.

(** * Definitions: Undirected Graph and without loop
  *)

Class NGraph `(Gr: Graph) := {
 gsym: symmetric Adj;
 grefl: irreflexive Adj}.

Section NGraph.

Generalizable Variables V Adj.
Context `(Gr: NGraph V Adj).

(** 
    Edges on V linked thanks to Adj
 *)

Definition E := (@edge_finType V Adj).

(** 
    VtoE v w e0: returns an edge made with v and w if they are adjacent
                         e0 otherwise
 *)
Definition VtoE (v w : V) (e0 : E) : E := 
 odflt e0 (insub (v, w)).


(** * Lemmas: Undirected Graph and without loop
  *)

(** **  VtoE
 *)

Lemma VtoE1 e e0 : VtoE (fste e) (snde e) e0 = e.
Proof.
rewrite /VtoE.
case: insubP; case=> /=;last first.
 by unfold fste,snde;rewrite (EdgeValP e).
move=> e' Pe1 Pe2 Pe3. apply edge_eq=>//=.
rewrite Pe3;unfold fste, snde. destruct e=>//=.
destruct edgeVal0=>//. 
Qed.    

Lemma VtoE2 : forall v w e0, Adj v w ->
 (enum_rank v < enum_rank w)%nat ->
 (fste (VtoE v w e0) == v)%B.
Proof. 
rewrite /VtoE=> v w e0 h1 h2.
case:insubP=>/= [u hu1 hu2 | hu];last first.
 by rewrite h1 h2 in hu.
move:hu2. unfold fste;destruct u=>/=. destruct edgeVal0=>/=. 
move=>H. injection H=>_ ->. by apply eq_refl. 
Qed. 

Lemma VtoE3 : forall v w e0, Adj v w ->
 (enum_rank v < enum_rank w)%nat ->
 (snde (VtoE v w e0) == w)%B.
Proof. 
rewrite /VtoE=> v w e0 h1 h2.
case:insubP=>/= [u hu1 hu2 | hu];last first.
 by rewrite h1 h2 in hu.
move:hu2. unfold snde;destruct u=>/=. destruct edgeVal0=>/=. 
move=>H. injection H=> -> _. by apply eq_refl. 
Qed.


(** **  deg 
 *)

Lemma deg_card : forall (v:V), 
 (deg Gr0 v <= #|V|.-1)%coq_nat.
Proof.
move=>v;case h:(deg Gr0 v <= #|V|.-1);first by move/leP:h.
rewrite deg_card1 in h=>//. 
apply grefl.
Qed.

(** **  Adj 
 *)

Lemma adj_diff : forall u v,
 Adj u v -> u != v.
Proof. 
move=>u v.
case h:(u == v)=>//.
move/eqP:h->.
by rewrite grefl.
Qed.

(** ** Port numbering
  *)

Variable nu : V -> seq V.
Hypothesis Hnu: forall (v w:V), (Adj v w) = (w \in (nu v)). 
Hypothesis Hnu2: forall (v :V), uniq (nu v). 

Lemma degnu1 : forall v, size (nu v) = deg Gr0 v.
Proof. 
move=>v.
unfold deg,Nb_enum.
apply/eqP. rewrite -uniq_size_uniq=>//.
 apply enum_uniq.
intro. rewrite -Hnu mem_enum. done. 
Qed. 

Lemma degnu3 : forall v w i,
 i < deg Gr0 v  -> nth v (nu v) i = w ->
 w \in (nu v).
Proof. 
setoid_rewrite <-degnu1=>v. 
elim:(nu v)=>//=;intros w l hind a i.
case:i=>//=.
 by intros;rewrite in_cons H0 eq_refl.
intros;rewrite in_cons (hind _ _ _ H0)=>//. 
by rewrite orbT.
Qed.

Lemma degnu2 : forall v w i,
 i < deg Gr0 v  -> nth v (nu v) i = w ->
 exists j, j < deg Gr0 w /\ nth w (nu w) j = v.
Proof.
intros. exists (index v (nu w)). 
 split;last first. 
 apply nth_index. rewrite -Hnu gsym Hnu. apply (degnu3 H H0).
rewrite -degnu1 index_mem.   rewrite -Hnu gsym Hnu. apply (degnu3 H H0). 
Qed. 

End NGraph. 


